#ifndef __PLATFORMMUTEX_H
#define __PLATFORMMUTEX_H

#include <sys/synchronization.h>
#include <cell/atomic.h>
#include <semaphore.h>
/**
 *	A platform/api specific mutex class. Always recursive (a single thread can lock multiple times).
 */

namespace UnityPlugin
{
	class NonCopyable
	{
	public:
		NonCopyable() {}

	private:
		NonCopyable(const NonCopyable&);
		NonCopyable& operator=(const NonCopyable&);
	};

	class PlatformMutex : public NonCopyable
	{
		friend class Mutex;
	protected:
		PlatformMutex();
		~PlatformMutex();

		void Lock();
		void Unlock();
		bool TryLock();

	private:

		sys_lwmutex_attribute_t lwmutexattr;
		sys_lwmutex lwmutex;
	};

	class PlatformSemaphore : public NonCopyable
	{
	public:
		PlatformSemaphore();
		~PlatformSemaphore();

		void Reset();
		void WaitForSignal();
		void Signal();

	private:
		void Create();
		void Destroy();

	private:
		sem_t	m_Semaphore;
	};

	// AtomicAdd - Returns the new value, after the operation has been performed (as defined by OSAtomicAdd32Barrier)
	inline int AtomicAdd (int volatile* i, int value) {
		return cellAtomicAdd32((uint32_t*)i, value) + value;	// on ps3 it returns the pre-increment value
	}

	// AtomicSub - Returns the new value, after the operation has been performed (as defined by OSAtomicSub32Barrier)
	inline int AtomicSub (int volatile* i, int value) {
		return cellAtomicSub32((uint32_t*)i, value) - value;	// on ps3 it returns the pre-increment value
	}

	// AtomicIncrement - Returns the new value, after the operation has been performed (as defined by OSAtomicAdd32Barrier)
	inline int AtomicIncrement (int volatile* i) {
		return cellAtomicIncr32((uint32_t*)i)+1;	// on ps3 it returns the pre-increment value
	}

	inline int AtomicDecrement (int volatile* i) {
		return cellAtomicDecr32((uint32_t*)i)-1;	// on ps3 it returns the pre-increment value
	}

	inline bool AtomicCompareExchange (int volatile* i, int newValue, int expectedValue) {
		return cellAtomicCompareAndSwap32((uint32_t*)i, (uint32_t)expectedValue, (uint32_t)newValue) == (uint32_t)expectedValue;
	}
}
#endif // __PLATFORMMUTEX_H
